// src/machine/machine.controller.ts
import {
  Controller,
  Get,
  Param,
  ParseIntPipe,
  NotFoundException,
} from '@nestjs/common';
import {
  ApiTags,
  ApiOperation,
  ApiResponse,
  ApiProduces,
  ApiParam,
} from '@nestjs/swagger';
import { MachineService } from './machine.service';

@Controller('machines')
@ApiTags('Machines') // Group under "Machines" in Swagger
export class MachineController {
  constructor(private machineService: MachineService) {}

  /**
   * Get a machine by its ID.
   * @param id The ID of the machine.
   * @returns The machine object.
   */
  @Get(':id')
  @ApiOperation({
    summary: 'Get a machine by ID',
    description: 'Retrieves a single machine using its unique ID.',
  })
  @ApiProduces('application/json')
  @ApiParam({
    name: 'id',
    type: Number,
    description: 'Unique identifier of the machine',
  })
  @ApiResponse({
    status: 200,
    description: 'Machine retrieved successfully.',
    schema: {
      example: {
        id: 1,
        name: 'CNC Router',
        description: 'Computer-controlled milling machine for precision parts.',
      },
    },
  })
  @ApiResponse({
    status: 404,
    description: 'Machine not found with the provided ID.',
  })
  async getMachineById(@Param('id', ParseIntPipe) id: number) {
    const machine = await this.machineService.getMachineById(id);
    if (!machine) {
      throw new NotFoundException('Machine not found');
    }
    return machine;
  }

  /**
   * Get a list of all machines.
   * @returns An array of machine objects.
   */
  @Get()
  @ApiOperation({
    summary: 'Get all machines',
    description: 'Retrieves a list of all available machines.',
  })
  @ApiProduces('application/json')
  @ApiResponse({
    status: 200,
    description: 'List of machines retrieved successfully.',
    schema: {
      example: [
        {
            id: 1,
            name: 'CNC Router',
            description: 'Computer-controlled milling machine for precision parts.',
        },
        {
            id: 2,
            name: '3D Printer',
            description: 'Fused deposition modeling printer for rapid prototyping.',
        },
      ],
    },
  })
  async getAllMachines() {
    return this.machineService.getAllMachines();
  }

  @Get(':id/node')
  @ApiOperation({
    summary: 'Get nodeId by machine ID',
    description: 'Retrieves the nodeId associated with the given machine ID.',
  })
  @ApiProduces('application/json')
  @ApiParam({
    name: 'id',
    type: Number,
    description: 'Unique identifier of the machine',
  })
  @ApiResponse({
    status: 200,
    description: 'nodeId retrieved successfully.',
    schema: {
      example: {
        nodeId: 'node-A1',
      },
    },
  })
  @ApiResponse({
    status: 404,
    description: 'Machine not found with the provided ID.',
  })
  async getMachineNodeId(@Param('id', ParseIntPipe) id: number) {
    const result = await this.machineService.getMachineNodeById(id);
    if (!result) {
      throw new NotFoundException('Machine not found');
    }
    return result;
  }

}
